%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function CMPSm2(parameterFile,coefficientFile,cellFileInitPt,startCell,endCell); 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%	Continuation Method for Polynomial Systems (MATLAB), 
%   Ver. 1.0 January, 2002
%   Ver. 1.1 October 26, 2002
%   Ver. 1.2 December 31, 2002
%   Ver. 2.0 April 14, 2003
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%
% Example 1: Linear homotopy for the cyclic-3 polynomial
%
% parameterFile='para_3cycL.dat'; coefficientFile='3cycL.dat2'; cellFileInitPt='3cycL'; 
%
%%%

%%%
% Example 2: Cheater's homotopy for the cyclic-3 polynomial
%
% parameterFile='para_3cyc.dat'; coefficientFile='3cyc.dat2'; cellFileInitPt='3cyc';
% startCell=1; endCell=2;
%
%%%

noOfInputArguments=nargin; 
%
%	= 3 for linear homotopy case;
%	= 5 for polyhedral and cheater's homotopy cases; 
%
if noOfInputArguments==3
	startCell = 1;
	endCell = 1;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
informationLevel = -1; % = -1 : default
%
%			=-1 -->	The simplest information, no *.note file
%			= 0 --> Some additional information on power of t, etc., and the minimum 
%					singular value in *.note file when it becomes very small
%			=+1 --> Detailed information on each iteration in *.note file
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
coeffSW = 1; % = +1 : default
%
% 			=+1 --> To read input cell data file and setting the random coefficients 
%					inputed for the coefficints which the program uses.
%			=-1 --> To read input cell data file and setting the minus of 
%					the random coefficients for the coefficients which 
%					the program uses.		
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
modSW = 0; % = 0 : default
%
% 			= 0 -->	Paths starting from all initial points are computed. 
% 			= 1 --> Only paths starting from initial points 1, 1+n, 1+2n, ... 
%					in each cell files are computed. 
%					This setting is for the type-2 symmetric structure involved 
%					in the cyclic polynomial problem 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% fprintf('\n'); 
% fprintf('     ##################################################\n');
% fprintf('     #   Continuation Method for Polynomial Systems   #\n');
% fprintf('     #       Sunyoung Kim and Masakazu Kojima         #\n');
% fprintf('     #     MATLAB, Version 1.0, December 23, 2001     #\n'); 
% fprintf('     #             Version 1.1,  October 26, 2002     #\n'); 
% fprintf('     #             Version 2.0,  February, 2003       #\n'); 
% fprintf('     ##################################################\n\n');

%%% Three output files --->
if noOfInputArguments == 5
	noteFileName = strcat(cellFileInitPt,num2str(startCell),'-',num2str(endCell),'.note');
else
	noteFileName = strcat(cellFileInitPt,'.note');
end	
%
if noOfInputArguments == 5
	solFileName  = strcat(cellFileInitPt,num2str(startCell),'-',num2str(endCell),'.sol');
else
	solFileName  = strcat(cellFileInitPt,'.sol');
end	
solFID = fopen(solFileName,'w'); 
%
if noOfInputArguments == 5
	statFileName = strcat(cellFileInitPt,num2str(startCell),'-',num2str(endCell),'.stat');
else
	statFileName = strcat(cellFileInitPt,'.stat');
end	
%%% <--- Three output file

writeHeadings(noOfInputArguments,1,coefficientFile,cellFileInitPt,parameterFile,...
	noteFileName,solFileName,statFileName,startCell,endCell,modSW,coeffSW); 

%%% Reading values for major parameters 
[statusSW,parameter] = readParameter(parameterFile);
if statusSW == -1 
	fprintf('\n*** No %s file or imcomplete %s file ***\n',parameterFile); disp(parameter); 
	return 
end

%%% Choosing a scaling factor beta for the homotopy parameter t
% [statusSW,aveMaxPower,maxMaxPower] = check2Power(startCell,cellFileInitPt);
% if statusSW == -1 
%	fprintf('\n*** No %s files starting from %s1 ***\n',cellFileInitPt,cellFileInitPt); 
%	return 
% else 
% 	parameter.beta = max([1, aveMaxPower/ 50]); 
% end
writeParameters(1,parameter); 

%%% Readiing data on the polynomial system to be solved
statusSW = 0; 
commonDataFileName = coefficientFile; % strcat(coefficientFile,'.dat2'); 
[statusSW,nDim,mVect,noTermsVect,cMatrix,coeffMatrix] ... 
	= readCommonData(coeffSW,commonDataFileName);
if statusSW == -1 
	fprintf('\n*** No %s file or imcomplete %s file ***\n',commonDataFileName); return 
end

noOfInputArguments=nargin; 

%%% writeHeading to the three output files --->
if informationLevel >= 0 
	noteFID = fopen(noteFileName,'w');
	fprintf(noteFID,'# %s \n',noteFileName); 
	writeHeadings(noOfInputArguments,noteFID,coefficientFile,cellFileInitPt,parameterFile,...
		noteFileName,solFileName,statFileName,startCell,endCell,modSW,coeffSW); 
	writeParameters(noteFID,parameter); 
	fclose(noteFID); 
end
%
solFID = fopen(solFileName,'w'); 
fprintf(solFID,'# %s \n',solFileName); 
writeHeadings(noOfInputArguments,solFID,coefficientFile,cellFileInitPt,parameterFile,...
	noteFileName,solFileName,statFileName,startCell,endCell,modSW,coeffSW); 
writeParameters(solFID,parameter); 
fclose(solFID); 
%
statFID = fopen(statFileName,'w');
fprintf(statFID,'# %s \n',statFileName); 
writeHeadings(noOfInputArguments,statFID,coefficientFile,cellFileInitPt,parameterFile,...
	noteFileName,solFileName,statFileName,startCell,endCell,modSW,coeffSW); 
writeParameters(statFID,parameter); 
writeExpParam(statFID,informationLevel);
fclose(statFID);
%%% <--- writeHeadings to the three output file

writeExpParam(1,informationLevel);

previousPrint = -21; 

for kcell=startCell:endCell
	if noOfInputArguments == 5
		dataFileName = strcat(cellFileInitPt,num2str(kcell));
	else
		dataFileName = cellFileInitPt;
	end		
	%%% Reading each cell data containing multiple initial points ---> 
	[statusSW,powerMatrix,numOfInitPoints,initSolutions] = ...
		readEachCellData(parameter.beta,coeffSW,nDim,mVect,noTermsVect,cMatrix,dataFileName); 
	if statusSW == -1
		fprintf('\n*** No %s file or imcomplete %s file ***\n',dataFileName); return, 
	end	
	%%% <--- Reading each cell data containing multiple initial points
	%%% for the type-2 symmetric structure involved in the cyclic polynomial problem ---> 
	if modSW == 1, stepInitPoints = nDim; else, stepInitPoints = 1; end
	%%% <--- for the type-2 symmetric structure involved in the cyclic polynomial problem
	if noOfInputArguments == 5
		outputHeadName = strcat(cellFileInitPt,num2str(kcell));
	else
		outputHeadName = cellFileInitPt;
	end	
	for pthInitPoint=1:stepInitPoints:numOfInitPoints
		previousPrint = previousPrint - 1; 
		%%% Setting the initial point in the complex space C^n and t = 0
		xSigma0Vect = real(initSolutions(pthInitPoint,:))' ... 
			+ imag(initSolutions(pthInitPoint,:))' * i;
		xSigma0Vect = [xSigma0Vect; -20.0];
		%%% Calling cheater's homotopy continuation method given a starting point ---> 
		[maxCondNo,CPUsecond,fSigma,hValError,maxNormOFx,maxSing,maxPower,minSing,mindSigmaRatio,normOFdx,normOFx,...
			normOFpDirect,predIteration,statusP,totalCorrIteration,xSigmaCurrent] ... 
			= traceOneCurve2(coeffMatrix,informationLevel,kcell,nDim,noTermsVect,noteFileName,...
				numOfInitPoints,outputHeadName,parameter,powerMatrix,pthInitPoint,statFileName,xSigma0Vect);
		%%% <--- Calling cheater's homotopy continuation method given a starting point
		%%% Adding statistical information on the problem just solved to the stat file ---> 
		statFID = fopen(statFileName,'a');
		[previousPrint] = writeStatistics(statFID,informationLevel,outputHeadName,previousPrint,modSW,kcell,... 
			pthInitPoint,statusP,predIteration,totalCorrIteration,fSigma,hValError,maxNormOFx,mindSigmaRatio,...
			normOFx,normOFdx,normOFpDirect,maxSing,minSing,maxPower,CPUsecond,maxCondNo);		
		fclose(statFID); 
		%%% <--- Adding statistical information on the problem just solved to the stat file 
		%%% Printing statistical information on the problem just solved ---> 
		[previousPrint] = writeStatistics(1,informationLevel,outputHeadName,previousPrint,modSW,kcell,... 
			pthInitPoint,statusP,predIteration,totalCorrIteration,fSigma,hValError,maxNormOFx,mindSigmaRatio,...
			normOFx,normOFdx,normOFpDirect,maxSing,minSing,maxPower,CPUsecond,maxCondNo);		
		%%% <--- Printing statistical information on the problem just solved	
		%%% Adding solution information on the problem just solved ---> 
		if statusP >= 2
			solFID = fopen(solFileName,'a');
			writeSolFile(solFID,nDim,xSigmaCurrent,hValError,kcell,pthInitPoint); 
			fclose(solFID); 
		end
		%%% <--- Adding solution information on the problem just solved 
		fclose('all'); 
    end	% " kk=1:stepInitPoints:numOfInitPoints"
end % " kcell=startCell:endCell"
fclose('all');
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,numericData] = checkAndRead(oneLine,keyWord,expectLength);
statusSW = 0;
lenKeyWord = length(keyWord);
lenOneLine = length(oneLine);
if lenOneLine <= lenKeyWord
	numericData = 0;
	return;
end
if keyWord(1) == '?'
	keyWord = keyWord(2:lenKeyWord);
	lenKeyWord = lenKeyWord-1;
	oneLine = oneLine(2:lenOneLine);
	lenOneLine = lenOneLine-1;
end	
if strcmp(keyWord,oneLine(1:lenKeyWord)) == 0
	numericData = 0;
	return;
else
	numericData = str2num(oneLine(lenKeyWord+1:lenOneLine));
%	disp(numericData); 
%	length(numericData)
	if length(numericData) ~= expectLength 
		statusSW = -1; 
	else
		statusSW = 1; 
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,oneLine] = getOneLine(dataFID);
flowCTRL = 0;
oneLine = '#'; 
while (feof(dataFID)==0) & (flowCTRL== 0)
 	oneLine = fgetl(dataFID);
	len = length(oneLine); 
	if (len > 0) & (oneLine(1)~='#')
		p=1;
		while (p<=len) & (oneLine(p)==' ') 
			p = p+1;
		end
		if p<=len
			flowCTRL=1;
		end
	end
end
if flowCTRL==0
	oneLine = '#';
	statusSW = -1; 
else
	statusSW = 1;
end
return 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,parameter] = readParameter(parameterFile);
paraFID = fopen(parameterFile,'r');
if paraFID < 0 
	statusSW = -1; parameter = 0; return;
else 
	statusSW = 0;
end

% Default parameter values ---> 
parameter.accINfVal = 1.e-10; 
parameter.accInNewtonDir= 1.e-8; 
parameter.beta=1; 
parameter.divMagOFx= 1.0e+3; 
parameter.dTauMax= 0.5; 
parameter.minEigForNonsing= 1.e-12; 
parameter.NewtonDirMax= 0.1; 
parameter.predItMax= 2000; 
% <--- Defaulat parameter values 

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
	[SW2,value] = checkAndRead(oneLine,'accINfVal=',1); % 
	if SW2 == 1
		parameter.accINfVal = value;
	else
		statusSW = -1;
	end
else
	statusSW = -1; 
end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
	[SW2,value] = checkAndRead(oneLine,'accInNewtonDir=',1); % 
	if SW2 == 1
		parameter.accInNewtonDir = value;
	else
		statusSW = -1;
	end
else
	statusSW = -1; 
end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
 	[SW2,value] = checkAndRead(oneLine,'beta=',1); % 
 	if SW2 == 1
		parameter.beta = value;
 	else
 		statusSW = -1;
 	end
 else
 	statusSW = -1; 
end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
	[SW2,value] = checkAndRead(oneLine,'divMagOFx=',1); % 
	if SW2 == 1
		parameter.divMagOFx = value;
	else
		statusSW = -1;
	end
else
	statusSW = -1; 
end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
	[SW2,value] = checkAndRead(oneLine,'dTauMax=',1); % 
	if SW2 == 1
		parameter.dTauMax = value;
	else
		statusSW = -1;
	end
else
	statusSW = -1; 
end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
	[SW2,value] = checkAndRead(oneLine,'minEigForNonsing=',1); % 
	if SW2 == 1
		parameter.minEigForNonsing = value;
		[SW,oneLine] = getOneLine(paraFID);
		if SW == 1
			[SW2,value] = checkAndRead(oneLine,'NewtonDirMax=',1); 
			if SW2 == 1
				parameter.NewtonDirMax = value;
			else
				statusSW = -1;
			end
		else
			statusSW = -1;
		end		
	else
		[SW2,value] = checkAndRead(oneLine,'NewtonDirMax=',1); 
		if SW2 == 1
			parameter.NewtonDirMax = value;
		else
			statusSW = -1;
		end
	end
else
	statusSW = -1; 
end

% [SW,oneLine] = getOneLine(paraFID);
% if SW == 1
% 	[SW2,value] = checkAndRead(oneLine,'NewtonDirMax=',1); 
% 	if SW2 == 1
%		parameter.NewtonDirMax = value;
%	else
%		statusSW = -1;
%	end
% else
% 	statusSW = -1; 
% end

[SW,oneLine] = getOneLine(paraFID);
if SW == 1
 	[SW2,value] = checkAndRead(oneLine,'predItMax=',1); % 
	if SW2 == 1
		parameter.predItMax = value;
	else
		statusSW = -1;
	end
else
 	statusSW = -1; 
end

fclose(paraFID);
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function writeHeadings(noOfInputArguments,outputFID,coefficientFile,cellFileInitPt,parameterFile,...
	noteFileName,solFileName,statFileName,startCell,endCell,modSW,coeffSW); 
fprintf(outputFID,'##################################################\n');
fprintf(outputFID,'#   Continuation Method for Polynomial Systems   #\n');
fprintf(outputFID,'#       Sunyoung Kim and Masakazu Kojima         #\n');
fprintf(outputFID,'#     MATLAB, Version 1.0, December 23, 2001     #\n'); 
fprintf(outputFID,'#             Version 1.1,  October 26, 2002     #\n'); 
fprintf(outputFID,'#             Version 2.0,  April 14, 2003       #\n'); 
fprintf(outputFID,'##################################################\n');
fprintf(outputFID,'#\n');
fprintf(outputFID,'### Input data ### \n');
fprintf(outputFID,'#\n');
fprintf(outputFID,'# coefficientFile=%s, ',coefficientFile);
if noOfInputArguments==5
	fprintf(outputFID,'cellFilesInitPt=%s%d-%d, ',cellFileInitPt,startCell,endCell);	
else
	fprintf(outputFID,'cellFileInitPt=%s, ',cellFileInitPt);
end
fprintf(outputFID,'parameterFile=%s\n#\n',parameterFile);
% fprintf(outputFID,'parameterFile=%s, ',parameterFile);
% fprintf(outputFID,'modSW=%+d, coeffSW=%+d\n#\n',modSW,coeffSW);
fprintf(outputFID,'### Output files ### \n');
fprintf(outputFID,'#\n');
fprintf(outputFID,'# Statistical information ------ %s\n',statFileName);
fprintf(outputFID,'# Solution information --------- %s\n',solFileName);
fprintf(outputFID,'# Emergency information --- %s\n',noteFileName);
fprintf(outputFID,'#\n');
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function writeExpParam(outputFID,informationLevel);
fprintf(outputFID,'### Notation ###\n');
fprintf(outputFID,'#\n');
fprintf(outputFID,'# cell --------  cell number, e.g., 8cyc25 file where cell=25\n'); 
fprintf(outputFID,'# init --------  initial point number in each cell, e.g., sol5 where init=5\n'); 
fprintf(outputFID,'# statusP -----  the status of the homotopy path\n');
fprintf(outputFID,'#                = -4 --- fail because of too large magnitude of an iterated point\n');
fprintf(outputFID,'#                = -3 --- fail because of too many predictor iterations\n');
fprintf(outputFID,'#                = -2 --- divergence\n');
fprintf(outputFID,'#                = -1 --- fail becasue of too small predictor step length\n'); 
fprintf(outputFID,'#                =  3 --- both accINfVal and accInNewtonDir are attained;\n');
fprintf(outputFID,'#                         a nonsingular solution\n');
fprintf(outputFID,'#                =  4 --- accINfVal is attained but not accInNewtonDir;\n');
fprintf(outputFID,'#                         a singular solution\n');
fprintf(outputFID,'#                =  5 --- neither accINfVal nor accInNewtonDir is attained;\n');
fprintf(outputFID,'#                         either a singular solution or divergence?\n');
fprintf(outputFID,'# pIt ---------  the total predictor iterations\n'); 
fprintf(outputFID,'# TcIT --------  the total corrector iterations\n'); 
fprintf(outputFID,'# cpu    ------- the cpu second required to compute the homotopy path\n'); 
fprintf(outputFID,'# normOFx -----  the 2 norm of the approximate solution\n'); 
fprintf(outputFID,'# hValError ---  the infinity norm function value error at the approximate solution\n');
fprintf(outputFID,'# normOFdx ----- the 2 norm of the Newton direction at the last Newton iteration\n');
fprintf(outputFID,'# minSing ------ the minimum singluar value of Df(x) at the approximate solution x,\n');
fprintf(outputFID,'#                where Df(x) denotes the Jacobian matrix\n'); 
% fprintf(outputFID,'# maxSing ------ the maximum singular value of Df(x)\n'); 
if informationLevel >= 0
	fprintf(outputFID,'# maxSing ------ the maximum singular value of Df(x)\n'); 
    fprintf(outputFID,'# finalS ------- the value of final sigma at which convergence is checked\n');
    fprintf(outputFID,'# mindSRatio --- the minimum of dSigma / (-sigma) over all iterations\n'); 
    fprintf(outputFID,'# maxNormOFx --- the maximum of the 2 norm of x over all iterates\n');
    fprintf(outputFID,'# maxCondNo ---- the maximum condition number of D_x h(x,Sigma) over all iterates\n');
    fprintf(outputFID,'# maxPower ----- the maximum power of t over all terms in the homotopy\n');
end
fprintf(outputFID,'#\n');
fprintf(outputFID,'###\n');
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function writeParameters(noteFID,parameter); 
% fprintf(noteFID,'#\n');
fprintf(noteFID,'### Major prameter values ###\n');
% fprintf(noteFID,'# %s \n',noteFileName);
fprintf(noteFID,'#\n');
fprintf(noteFID,'# accINfVal = %6.2e, ',parameter.accINfVal); 
fprintf(noteFID,'accInNewtonDir = %6.2e, ',parameter.accInNewtonDir); 
fprintf(noteFID,'beta         = %6.2e\n',parameter.beta); 
fprintf(noteFID,'# divMagOFx = %6.2e, ',parameter.divMagOFx); 
fprintf(noteFID,'dTauMax        = %6.2e, ',parameter.dTauMax); 
% fprintf(noteFID,'minEigForNonsing=%6.2e\n',parameter.minEigForNonsing); 
fprintf(noteFID,'NewtonDirMax = %6.2e\n',parameter.NewtonDirMax); 
fprintf(noteFID,'# predItMax = %d\n#\n',parameter.predItMax); 
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [previousPrint] = writeStatistics(outputFID,informationLevel,outputHeadName,previousPrint,...
	modSW,kcell,pthInitPoint,statusP,predIteration,totalCorrIteration,fSigma,hValError,maxNormOFx,...
	mindSigmaRatio,normOFx,normOFdx,normOFpDirect,maxSing,minSing,maxPower,CPUsecond,maxCondNo); 
if (previousPrint <= -20) & ( (pthInitPoint == 1) | (modSW == 0) ),
	if (outputFID == 1)
		fprintf(outputFID,'%s\n',outputHeadName); 
		previousPrint = 0; 
	end
	if informationLevel >= 0
%		fprintf(outputFID,'#   cell init statusP  pIT   TcIT hValError minPstep  normOFx normOFdx   maxSing    minSing   maxPower    cpu\n');
%		fprintf(outputFID,'#   cell init statusP  pIT   TcIT hValError minPstep    normOFx      minSing    cpu normOFdx   maxSing   maxPower\n'); 
		fprintf(outputFID,'#  cell  init statusP   pIT    TcIT    cpu    normOFx hValError normOFdx  minSing  maxSing    finalS mindSRatio maxNormOFx maxCondNo   maxPower\n'); 
	else
%		fprintf(outputFID,'#  cell  init statusP   pIT    TcIT    cpu    fSigma hValError normOFdx     normOFx minSing\n'); 
%        fprintf(outputFID,'#  cell  init statusP   pIT    TcIT    cpu    normOFx hValError normOFdx  minSing  maxSing\n');
        fprintf(outputFID,'#  cell  init statusP   pIT    TcIT    cpu    normOFx hValError normOFdx  minSing\n');
	end
end
if informationLevel >= 0
	fprintf(outputFID,'%7d  %4d  %+2d    %6d %7d %6.2f %8.4e  %6.2e %6.2e %6.2e %6.2e %6.2e   %6.2e   %6.2e  %6.2e %8.4e\n',...
		kcell,pthInitPoint,statusP,predIteration,totalCorrIteration,CPUsecond,normOFx,hValError,...
		normOFdx,minSing,maxSing,fSigma,mindSigmaRatio,maxNormOFx,maxCondNo,maxPower); 
else
%	fprintf(outputFID,'%7d  %4d  %+2d    %6d %7d %6.2f %8.4e  %6.2e %6.2e %6.2e %6.2e\n',...
%		kcell,pthInitPoint,statusP,predIteration,totalCorrIteration,CPUsecond,normOFx,hValError,...
%		normOFdx,minSing,maxSing); 
	fprintf(outputFID,'%7d  %4d  %+2d    %6d %7d %6.2f %8.4e  %6.2e %6.2e %6.2e\n',...
		kcell,pthInitPoint,statusP,predIteration,totalCorrIteration,CPUsecond,normOFx,hValError,...
		normOFdx,minSing); 
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function writeSolFile(solFID,nDim,xSigmaCurrent,hValError,kcell,pthInitPoint); 
solution = zeros(1,2*nDim+3); 
for jthrow=1:nDim
	solution(1,(jthrow-1)*2+1) = real(xSigmaCurrent(jthrow,1));
	solution(1,(jthrow-1)*2+2) = imag(xSigmaCurrent(jthrow,1));
end
solution(1,2*nDim+1) = hValError;
solution(1,2*nDim+2) = kcell;
solution(1,2*nDim+3) = pthInitPoint; 

for k=1:nDim*2
	fprintf(solFID,'%+21.15e  ',solution(1,k));
end
fprintf(solFID,'%+21.15e  %10d  %4d\n',...
	solution(1,2*nDim+1),solution(1,2*nDim+2),solution(1,2*nDim+3));
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,aveMaxPower,maxMaxPower] = check2Power(startCell,fileHead);

dataFileName = strcat(fileHead,num2str(startCell));
maxMaxPower = 0.0; 
dataFID = fopen(dataFileName,'r');
if dataFID < 0
%	fclose(dataFID); 
	aveMaxPower = 0; maxMaxPower = 0; statusSW = -1; return
else
	statusSW = 1; 
	while feof(dataFID) == 0
		oneLine = fgetl(dataFID); 
		lenOneLine = length(oneLine); 
		if (lenOneLine >= 3) & (oneLine(1:2) == '= ')
			numLine = str2num(oneLine(3:lenOneLine));
			maxPower = max(numLine); 
			maxMaxPower = max([maxMaxPower,maxPower]); 
		end
	end
	fclose(dataFID); 
end

% return

if maxMaxPower == 0
	aveMaxPower = 0; 
	return
end
	
kCell = 0;
endCell = 100;

maxMaxPower = 0.0; 
totalMaxPower = 0.0;

continueSW = 1;
while continueSW == 1
	kCell = kCell + 1;
	dataFileName = strcat(fileHead,num2str(kCell));
	dataFID = fopen(dataFileName,'r');
	if dataFID < 0 
% 		fclose(dataFID); 
		kCell = kCell - 1; 
		continueSW = -1;
	else
		newMaxPower = 0; 
		while feof(dataFID) == 0
			oneLine = fgetl(dataFID); 
			lenOneLine = length(oneLine);  
			if (lenOneLine >= 3) & (oneLine(1:2) == '= ')
				numLine = str2num(oneLine(3:lenOneLine));
				maxPower = max(numLine); 
				newMaxPower = max([newMaxPower,maxPower]); 
			end
		end
		fclose(dataFID); 
		maxMaxPower = max([newMaxPower,maxMaxPower]);
		totalMaxPower = totalMaxPower + newMaxPower;
		if kCell >= endCell 
			continueSW = -1;
		end
	end
end

if kCell >= 1
	aveMaxPower = totalMaxPower / kCell;
	statsuSW = 1;
else
	maxMaxPower = 0; 
	aveMaxPower = 0;
	statusSW = -1;
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,nDim,mVect,noTermsVect,cMatrix,coeffMatrix] = ... 
	readCommonData(coeffSW,commonDataFileName);

inputFileID = fopen(commonDataFileName,'r');
if inputFileID < 0, 
	statusSW = -1; nDim = 0; mVect = 0; noTermVect = 0; cMatrix = 0; coeffMatrix = 0; 
	fclose('all'); 
	return;
else
	statusSW = 0;
end

flowCTRL = 1;
while (feof(inputFileID) ==0) & flowCTRL >= 0
	if flowCTRL == 1 
		[statusSW,nDim] = readnDim(inputFileID); 
		if statusSW == 1 
			flowCTRL = 2; 
		else flowCTRL = -1; end
	elseif flowCTRL == 2
		[statusSW,mVect] = readmVect(inputFileID,nDim);
		if statusSW == 1 
			flowCTRL = 3; 
		else flowCTRL = -1; end
	elseif flowCTRL == 3
		[statusSW,cMatrix] = readcMatrix(nDim,mVect,inputFileID); 
		if statusSW == 1 
			flowCTRL = 4;
		else flowCTRL = -1; end
	elseif flowCTRL == 4
		[statusSW,OcoefMatrix] = readOcoef(nDim,mVect,inputFileID);
		if statusSW == 1 
			flowCTRL = 5;
		else flowCTRL = -1; end
	elseif flowCTRL == 5
		[statusSW,RcoefMatrix] ... 
			= readRcoef(coeffSW,nDim,mVect,inputFileID);
		if statusSW == 1 
			flowCTRL = -2;
		else flowCTRL = -1; end
	end
end

if flowCTRL == -2
	statusSW = 0;
	noTermsVect = 2*mVect;
	for jthrow = 1:nDim
		coeffMatrix{jthrow} ... 
			= [RcoefMatrix{jthrow}, OcoefMatrix{jthrow}-RcoefMatrix{jthrow}];
	 	debug = 0;
		if debug == 1 
			fprintf('coeffMatrix[%2d] = ',jthrow);
			disp(coeffMatrix{jthrow});
		end
	end
else
	statusSW = -1; 
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,nDim] = readnDim(inputFileID);
debug = 0; 
keyWord = 'n = ';
expectLength = 1; 
statusSW = 0; 
while (statusSW == 0) & (feof(inputFileID) == 0)
	oneLine = fgetl(inputFileID); 
	[statusSW,nDim] = checkAndRead(oneLine,keyWord,expectLength);
end
if debug == 1
	fprintf('nDim = %d\n',nDim);
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,mVect] = readmVect(inputFileID,nDim);
debug = 0; 
keyWord = 'm = ';
expectLength = nDim; 
statusSW = 0; 
while (statusSW == 0) & (feof(inputFileID) == 0)
	oneLine = fgetl(inputFileID); 
	[statusSW,mVect] = checkAndRead(oneLine,keyWord,expectLength);
end
if debug == 1
	fprintf('mVect(1,:) = ');
	for j=1:nDim
		fprintf('%2d',mVect(1,j));
	end
	fprintf('\n');
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,cMatrix] = readcMatrix(nDim,mVect,inputFileID);
debug = 0; 
for j=1:nDim
	cMatrix{j} = sparse(zeros(mVect(1,j),nDim));
	for k=1:mVect(1,j)
		keyWord = strcat('?',num2str(j),'.',num2str(k),' = '); 
		statusSW = 0;
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			[statusSW,cMatrix{j}(k,:)] = checkAndRead(oneLine,keyWord,nDim);
		end
	end
end
if debug == 1
	for j=1:nDim
		for k=1:mVect(1,j)
			fprintf('cMatrix{%2d}(%2d,:) = ',j,k); 
				for p=1:nDim
					fprintf('%3d',cMatrix{j}(k,p));
				end
			fprintf('\n');
		end
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,OcoefMatrix] = readOcoef(nDim,mVect,inputFileID);
debug = 0; 
imagUnit = i;
for j=1:nDim
	OcoefMatrix{j} = zeros(1,mVect(1,j));
	for k=1:mVect(1,j)
		keyWordr = strcat('coef',num2str(j),'.',num2str(k),'r = ');
		keyWordi = strcat('coef',num2str(j),'.',num2str(k),'i = ');
		statusSW = 0;
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			[statusSW,realPart] = checkAndRead(oneLine,keyWordr,1);
		end
		statusSW = 0;
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			[statusSW,imagPart] = checkAndRead(oneLine,keyWordi,1);
		end
		OcoefMatrix{j}(1,k) = realPart + imagUnit * imagPart;
	end
end
if debug == 1 
	for j=1:nDim
		for k=1:mVect(1,j)
			fprintf('OcoefMatrix{%2d}(1,%2d) = ',j,k); 
			fprintf('%+20.15f%+20.15fi\n',real(OcoefMatrix{j}(1,k)),...
				imag(OcoefMatrix{j}(1,k)));
		end
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,RcoefMatrix] ...
	= readRcoef(coeffSW,nDim,mVect,inputFileID);
debug = 0; 
imagUnit = i;
for j=1:nDim
	RcoefMatrix{j} = zeros(1,mVect(1,j));
	for k=1:mVect(1,j)		
		keyWordr = strcat('Rcoef',num2str(j),'.',num2str(k),'r = ');
		keyWordi = strcat('Rcoef',num2str(j),'.',num2str(k),'i = ');
		statusSW = 0;
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			[statusSW,realPart] = checkAndRead(oneLine,keyWordr,1);
%			realPart
		end
		statusSW = 0;
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			[statusSW,imgPart] = checkAndRead(oneLine,keyWordi,1);
%			imgPart
		end
		if coeffSW == 1 
			RcoefMatrix{j}(1,k) = realPart + imgPart * imagUnit;
		else % coeffSW == -1
			RcoefMatrix{j}(1,k) = -realPart - imgPart * imagUnit;
		end
%
	end
end
if debug == 1 
	for j=1:nDim
		for k=1:mVect(1,j)
			fprintf('RcoefMatrix{%2d}(1,%2d) = ',j,k); 
			fprintf('%+20.15f%+20.15fi\n',real(RcoefMatrix{j}(1,k)),...
				imag(RcoefMatrix{j}(1,k)));
		end
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,powerMatrix,numOfInitPoints,initSolutions] ... 
	= readEachCellData(beta,coeffSW,nDim,mVect,noTermsVect,cMatrix,dataFileName); 
inputFileID = fopen(dataFileName,'r');
if inputFileID < 0
	statusSW = -1; fclose('all'); return
else 
	statusSW = 0;
end
debug = 0; 

flowCTRL = 6;
while (feof(inputFileID) ==0) & flowCTRL >= 0
	if flowCTRL == 6
		[statusSW,numOfInitPoints] = ... 
			readnumOfInitPoints(inputFileID);
		if statusSW == 1 
			flowCTRL = 7; 
		else flowCTRL = -1; end
	elseif flowCTRL == 7
		[statusSW,initSolutions] = ... 
			readInitSols(nDim,numOfInitPoints,inputFileID);
		if statusSW == 1 
			flowCTRL = 8; 
		else flowCTRL = -1; end
	elseif flowCTRL == 8
		[statusSW,tPowerVect,minPower] = ...
			readtPowerVect(nDim,mVect,inputFileID);
 		flowCTRL = -1; 
	end
end

gammaP = min([beta,minPower]); 

for jthrow = 1:nDim
	powerMatrix{jthrow} = sparse(zeros(noTermsVect(jthrow),nDim+1));
	for kthterm = 1:mVect(1,jthrow)
		powerMatrix{jthrow}(kthterm,:) = ...
			[cMatrix{jthrow}(kthterm,:),tPowerVect{jthrow}(1,kthterm)/gammaP];
		powerMatrix{jthrow}(mVect(1,jthrow)+kthterm,:) = ...
			[cMatrix{jthrow}(kthterm,:),(tPowerVect{jthrow}(1,kthterm)+beta)/gammaP];
	end
end

if debug == 1
	for jthrow = 1:nDim
		for kthterm = 1:noTermsVect(1,jthrow)
			fprintf('powerMatrix{%2d}(%2d,:) = ',jthrow,kthterm);
			disp(powerMatrix{jthrow}(kthterm,:));
		end
	end
end

return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,numOfInitPoints] = readnumOfInitPoints(inputFileID);
debug = 0; 
imagUnit = i; 
keyWord = 'NumOfSols = ';
expectLength = 1; 
statusSW = 0; 
while (statusSW == 0) & (feof(inputFileID) == 0)
	oneLine = fgetl(inputFileID); 
	[statusSW,numOfInitPoints] = checkAndRead(oneLine,keyWord,expectLength);
end
if debug == 1 
	fprintf('numOfInitPoints = %d\n',numOfInitPoints);
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,initSolutions] ... 
	= readInitSols(nDim,numOfInitPoints,inputFileID);
debug = 0; 
imagUnit = i; 
initSolutions = zeros(numOfInitPoints,nDim);

statusSW = 0; 
while (statusSW == 0) & (feof(inputFileID) == 0)
	oneLine = fgetl(inputFileID);  
	lenOneLine = length(oneLine);
	if (lenOneLine >= 8) & (strcmp('sol',oneLine(1:3)) == 1)
		statusSW = 1;
	end
end
startingNumber = str2num(oneLine(4)); 
realPart = str2num(oneLine(8:lenOneLine)); 
oneLine = fgetl(inputFileID); %
imgPart = str2num(oneLine);
initSolutions(1,1) = realPart + imgPart * imagUnit;
for k=2:nDim
	statusSW = 0; 
	while (statusSW == 0) & (feof(inputFileID) == 0)
		oneLine = fgetl(inputFileID); 
		if (length(oneLine) > 1) & (oneLine(1) == '-' | oneLine(1) == '+' ... 
			| (47 < double(oneLine(1)) & double(oneLine(2)) < 58)) 
			realPart = str2num(oneLine);
			statusSW = 1;
		end
	end
	oneLine = fgetl(inputFileID); 
	imgPart = str2num(oneLine);
	initSolutions(1,k) = realPart + imgPart * imagUnit;
end

for j=1:numOfInitPoints-1
	keyWord = strcat('sol',num2str(j+startingNumber),' = '); % 
	statusSW = 0; 
	while (statusSW == 0) & (feof(inputFileID) == 0)
		oneLine = fgetl(inputFileID);  
		[statusSW,realPart] = checkAndRead(oneLine,keyWord,1);
	end
	oneLine = fgetl(inputFileID); %
	imgPart = str2num(oneLine);
	initSolutions(j+1,1) = realPart + imgPart * imagUnit;
	for k=2:nDim
		statusSW = 0; 
		while (statusSW == 0) & (feof(inputFileID) == 0)
			oneLine = fgetl(inputFileID); 
			if (length(oneLine) > 1) & (oneLine(1) == '-' | oneLine(1) == '+' ... 
				| (47 < double(oneLine(1)) & double(oneLine(2)) < 58)) 
				realPart = str2num(oneLine);
				statusSW = 1;
			end
		end
		oneLine = fgetl(inputFileID); 
		imgPart = str2num(oneLine);
		initSolutions(j+1,k) = realPart + imgPart * imagUnit;
	end
end
if debug == 1 
	for j=1:numOfInitPoints
		fprintf('initSolutions(%2d,:)\n',j); 
		for k=1:nDim
			fprintf('   %+20.15f%+20.15fi\n',real(initSolutions(j,k)),... 
				imag(initSolutions(j,k)));
		end
		fprintf('\n');
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [statusSW,tPowerVect,minPower] = ...
	readtPowerVect(nDim,mVect,inputFileID);
debug = 0; 
keyWord = strcat('= '); 
for j=1:nDim
	statusSW = 0; 
	while (statusSW == 0) & (feof(inputFileID) == 0)
		oneLine = fgetl(inputFileID);  
		[statusSW,tPowerVect{j}] = checkAndRead(oneLine,keyWord,mVect(1,j));
	end
%	mVect(1,j)
%	size(tPowerVect{j})
%	disp(tPowerVect{j})
end

minPower = 1.0e100; 
for j=1:nDim
	for k=1:mVect(1,j)
		if tPowerVect{j}(1,k) > 1.0e-50; 
			minPower = min([minPower, tPowerVect{j}(1,k)]);
		end
	end
end

if debug == 1 
	for j=1:nDim
		fprintf('tPowerVect{%2d}(1,:) = ',j); 
		for k=1:mVect(1,j)
			fprintf('%9.2f',tPowerVect{j}(1,k));
		end
		fprintf('\n');
	end
end
return
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
